# PropMaster.io Deployment Guide for cPanel Shared Hosting

## Prerequisites
- NameCheap shared hosting with cPanel
- Node.js support enabled
- MySQL database
- SSH access (optional but recommended)

## Step 1: Prepare Your Application

### 1.1 Build the Application
```bash
# Install dependencies
npm install

# Build for production (this creates the .next folder with compiled JS)
npm run build
```

**Important Notes:**
- The `npm run build` command transpiles TypeScript to JavaScript and creates the `.next` folder
- The `.next` folder contains the compiled JavaScript that cPanel can run
- Do NOT upload `node_modules` - cPanel will install dependencies
- Do NOT upload `.ts` source files - only upload compiled `.next` folder

### 1.2 Environment Configuration
Create a `.env` file in your project root with production values:
```env
NODE_ENV=production
DATABASE_URL="mysql://username:password@localhost:3306/database_name"
NEXTAUTH_URL="https://yourdomain.com"
NEXTAUTH_SECRET="your-secret-key"
HOSTNAME="yourdomain.com"
PORT=3000
```

## Step 2: Upload Files to cPanel

### 2.1 Upload via File Manager or FTP
- Upload the entire project folder to your public_html directory
- Or create a subdomain directory (recommended)

### 2.2 File Structure After Upload
```
public_html/
├── .htaccess (✅ Apache routing)
├── ecosystem.config.js (✅ PM2 config)
├── next.config.js (✅ Next.js config - standalone mode)
├── server.js (✅ Production server - JavaScript entry point)
├── package.json (✅ Dependencies)
├── package-lock.json (✅ Lock file)
├── node_modules/ (❌ Do NOT upload - cPanel installs this)
├── .next/ (✅ Compiled JavaScript - upload this!)
├── prisma/ (✅ Database schema)
├── src/ (❌ Do NOT upload .ts files - only compiled JS)
├── public/ (✅ Static assets)
├── setup-db.js (✅ Database setup)
├── run-setup.php (✅ PHP fallback)
└── DEPLOYMENT_README.md (✅ This guide)
```

**Critical Notes:**
- ✅ **Upload `.next/` folder** - Contains compiled JavaScript
- ❌ **Do NOT upload `node_modules/`** - cPanel installs dependencies
- ❌ **Do NOT upload `src/` with .ts files** - Only compiled code
- ✅ **Upload `server.js`** - This is your JavaScript entry point

## Step 3: Database Setup

### 3.1 Create MySQL Database
1. Go to cPanel → MySQL Databases
2. Create a new database
3. Create a database user
4. Assign user to database with all privileges

### 3.2 Run Database Migrations

**Option A: Via cPanel Node.js App (Recommended)**
1. Go to **cPanel → Setup Node.js App**
2. Select your application
3. In the **"Run NPM Script"** section, enter: `npx prisma generate && npx prisma db push && npm run db:seed`
4. Click **Run Script**

**Important Notes:**
- Use `npx prisma` instead of just `prisma` as Prisma CLI may not be globally installed
- If you get "command not found" errors, try running each command separately:
  1. `npx prisma generate`
  2. `npx prisma db push`
  3. `npm run db:seed`
- Make sure your DATABASE_URL environment variable is set correctly in cPanel

**Option B: Via File Manager (Alternative)**
1. Go to **cPanel → File Manager**
2. Navigate to your application directory
3. Create a file named `setup-db.js` with this content:
```javascript
const { exec } = require('child_process');

console.log('Setting up database...');

exec('npx prisma generate', (err, stdout, stderr) => {
  if (err) {
    console.error('Error generating Prisma client:', err);
    return;
  }
  console.log('Prisma client generated');

  exec('npx prisma db push', (err, stdout, stderr) => {
    if (err) {
      console.error('Error pushing database schema:', err);
      return;
    }
    console.log('Database schema pushed');

    exec('npm run db:seed', (err, stdout, stderr) => {
      if (err) {
        console.error('Error seeding database:', err);
        return;
      }
      console.log('Database seeded successfully');
    });
  });
});
```
4. Save the file and run it via Node.js app

**Option C: Via phpMyAdmin (Manual)**
If the above methods don't work:
1. Go to **cPanel → phpMyAdmin**
2. Import the SQL schema from `prisma/migrations/`
3. Run the seed data manually

## Step 4: Configure cPanel

### 4.1 Node.js Application Setup
1. Go to cPanel → Setup Node.js App
2. Create a new application:
    - **Node.js version:** 18.x or higher (✅ NameCheap supports this)
    - **Application mode:** Production
    - **Application root:** `/home/username/public_html`
    - **Application URL:** `https://yourdomain.com`
    - **Startup file:** `server.js` (✅ This is the compiled JavaScript entry point)

### 4.2 Environment Variables
Set environment variables in cPanel Node.js app settings:
- `NODE_ENV=production`
- `DATABASE_URL=your_database_url`
- `NEXTAUTH_URL=https://yourdomain.com`
- `NEXTAUTH_SECRET=your_secret`
- `HOSTNAME=yourdomain.com`
- `PORT=3000`

### 4.3 Install Dependencies
In cPanel Node.js app:
1. Click "Run NPM Install"
2. Wait for completion (may take 5-10 minutes on shared hosting)

**Important Notes:**
- Make sure Node.js version is 18+ (check with `node --version`)
- If you get "Cannot find module" errors, try running `npm install` again
- The installation may take several minutes on shared hosting
- Check the cPanel error logs if installation fails
- Some packages may show deprecation warnings - these are harmless

## Step 5: Start the Application

### 5.1 Start via cPanel
1. In Node.js app settings, click "Start App"
2. Check the application logs for any errors

### 5.2 Alternative: PM2 (if available)
```bash
# Via SSH
cd public_html
npm install -g pm2
pm2 start ecosystem.config.js --env production
pm2 save
pm2 startup
```

## Step 6: Configure Domain/Routing

### 6.1 .htaccess Configuration
The `.htaccess` file handles:
- HTTPS redirection
- Next.js routing
- Security headers
- Compression
- Caching

### 6.2 Health Check
Your app includes a `/health` endpoint for monitoring.

## Troubleshooting

### 503 Service Unavailable
1. Check Node.js app logs in cPanel
2. Verify environment variables
3. Check database connection
4. Ensure all dependencies are installed
5. Verify file permissions

### Cannot find module 'next'
This error occurs when `npm install` hasn't completed successfully:
1. Go to cPanel → Setup Node.js App
2. Click "Run NPM Install" again
3. Wait for completion (check logs)
4. If it fails, try running `npm install` manually via SSH
5. Ensure Node.js version is 18+ and compatible with Next.js 14

### Prisma command not found
This error occurs when Prisma CLI is not available:
1. Use `npx prisma` instead of `prisma` in commands
2. Try running commands separately:
   - `npx prisma generate`
   - `npx prisma db push`
   - `npm run db:seed`
3. If still failing, check if Prisma is installed: `npm list @prisma/client`
4. Reinstall if needed: `npm install @prisma/client prisma`

### TypeScript files not supported
NameCheap shared hosting does not run TypeScript directly:
1. ✅ **You MUST run `npm run build` locally first** - This creates the `.next` folder
2. ✅ **Upload the `.next` folder** - Contains compiled JavaScript
3. ❌ **Do NOT upload `src/` folder** - Contains TypeScript source files
4. ✅ **Set startup file to `server.js`** - This is the compiled entry point
5. The build process transpiles all `.ts` files to `.js` in the `.next` folder

### Node binary path warnings
The warning about node binary paths is harmless but can be resolved:
1. The warning won't prevent deployment
2. For production, these warnings can be ignored
3. If you want to fix it, use: `npm run --scripts-prepend-node-path`

### Deprecated Package Warnings
If you see npm warnings about deprecated packages:
1. The warnings are harmless and won't prevent deployment
2. For production builds, these warnings can be ignored
3. If you want to update packages, run:
   ```bash
   npm audit fix
   npm update
   ```
4. Some packages (like `abab`, `domexception`) are dev dependencies and won't affect production

### Database Connection Issues
1. Confirm DATABASE_URL format
2. Check MySQL user privileges
3. Verify database exists
4. Test connection locally

### Build Issues
1. Clear .next folder: `rm -rf .next`
2. Rebuild: `npm run build`
3. Check Node.js version compatibility

### Memory Issues
1. Increase memory limit in ecosystem.config.js
2. Monitor memory usage
3. Consider upgrading hosting plan

## Performance Optimization

### 1. Enable Compression
Already configured in .htaccess

### 2. Static Asset Caching
Configured for images, CSS, JS files

### 3. Database Optimization
- Use connection pooling
- Implement proper indexing
- Monitor slow queries

### 4. CDN (Optional)
Consider using a CDN for static assets

## Monitoring

### Logs
- Application logs: `logs/` directory
- cPanel Node.js app logs
- Database logs

### Health Checks
- `/health` endpoint returns "OK" when healthy
- Monitor response times
- Set up uptime monitoring

## Security Considerations

### 1. Environment Variables
- Never commit .env files
- Use strong secrets
- Rotate secrets regularly

### 2. File Permissions
- Set appropriate permissions on sensitive files
- Restrict access to .env and config files

### 3. HTTPS
- Always use HTTPS (configured in .htaccess)
- Consider SSL certificate renewal

### 4. Database Security
- Use strong passwords
- Limit database user privileges
- Regular backups

## Backup Strategy

### 1. Database Backups
```bash
# Automated backup script
npm run backup:create
```

### 2. File Backups
- Regular cPanel backups
- Version control for code
- Backup uploaded files

## Support

If you encounter issues:
1. Check application logs
2. Verify configuration
3. Test locally first
4. Contact hosting support if needed

## Quick Commands Reference

```bash
# Build and deploy
npm install
npm run build
npm start

# Database operations
npx prisma generate
npx prisma db push
npm run db:seed

# Logs
tail -f logs/combined.log
tail -f logs/err.log

# PM2 management (if using)
pm2 restart propmaster-io
pm2 logs propmaster-io
pm2 monit